<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

use App\Http\Services\DBClientConnection;

use App\Models\clientes\InstrumentosAudPamec;
use App\Models\clientes\HistoricoInsAudPamec;

class InstrumentosAudPamecController extends Controller
{
    private $cliente;

    function __construct() {
        // Crear la conexion temporal al esquema del cliente
        $this->middleware(function ($request, $next) {
            if (Auth::user() !== null) {
                $this->cliente = Auth::user()->cli_fk_id;
                $connection = new DBClientConnection();
                config(['database.connections.'.$this->cliente => $connection->getConnectionArray($this->cliente)]);
            }

            return $next($request);
        });
    }

    public function obtenerInstrumentoActivo(Request $request) {
        return InstrumentosAudPamec::on($this->cliente)
                                   ->selectRaw('iap_pk_id, iap_version, iap_es_acreditacion, iap_comentarios, iap_instrumento')
                                   ->where('iap_activa', true)
                                   ->orderBy('iap_es_acreditacion')
                                   ->get()->toArray();
    }

    public function obtenerInstrumentosConAcreditacion(Request $request) {
        return InstrumentosAudPamec::on($this->cliente)
                                    ->where('iap_es_acreditacion', true)
                                    ->orderBy('iap_activa', 'desc')
                                    ->orderBy('iap_pk_id', 'desc')
                                    ->get()->toArray();
    }

    public function obtenerInstrumentosSinAcreditacion(Request $request) {
        return InstrumentosAudPamec::on($this->cliente)
                                    ->where('iap_es_acreditacion', false)
                                    ->orderBy('iap_activa', 'desc')
                                    ->orderBy('iap_pk_id', 'desc')
                                    ->get()->toArray();
    }

    public function obtenerInstrumentoPorId(Request $request, $idVersion) {
        return InstrumentosAudPamec::on($this->cliente)
                                    ->where('iap_pk_id', $idVersion)
                                    ->get()->toArray();
    }

    public function crearInstrumento(Request $request) {
        $id = null;
        $nuevaVersion = null;

        DB::connection($this->cliente)->transaction(function() use($request, &$id, &$nuevaVersion) {
            $nuevaVersion = DB::connection($this->cliente)->select(
                $request->iap_es_acreditacion ? 'select nextval(\'"'.$this->cliente.'".seq_aud_pamec_con_acreditacion\')'
                                              : 'select nextval(\'"'.$this->cliente.'".seq_aud_pamec_sin_acreditacion\')'
            )[0]->nextval;

            $usuario = Auth::user()->id;

            $id = InstrumentosAudPamec::on($this->cliente)->create([
                'iap_version' => $nuevaVersion,
                'iap_es_acreditacion' => $request->iap_es_acreditacion,
                'iap_comentarios' => $request->iap_comentarios,
                'iap_instrumento' => json_encode($request->iap_instrumento),
                'iap_usu_creacion' => $usuario
            ])->iap_pk_id;

            HistoricoInsAudPamec::on($this->cliente)->create([
                'iap_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hap_accion' => 'Creación de una nueva versión del instrumento'
            ]);
        });

        return [$id, $nuevaVersion];
    }

    public function actualizarInstrumento(Request $request, $id) {
        DB::connection($this->cliente)->transaction(function() use($request, &$id) {
            $usuario = Auth::user()->id;

            InstrumentosAudPamec::on($this->cliente)->where('iap_pk_id', $id)->update([
                'iap_comentarios' => $request->iap_comentarios,
                'iap_instrumento' => $request->iap_instrumento
            ]);

            HistoricoInsAudPamec::on($this->cliente)->create([
                'iap_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hap_accion' => 'Actualización del instrumento'
            ]);
        });
    }

    public function activarInstrumento(Request $request, $id) {
        DB::connection($this->cliente)->transaction(function() use($request, &$id) {
            $usuario = Auth::user()->id;
            $esAcreditacion = $request->iap_es_acreditacion;

            $idActivo = InstrumentosAudPamec::on($this->cliente)
                                            ->where('iap_es_acreditacion', $esAcreditacion)
                                            ->where('iap_activa', true)
                                            ->pluck('iap_pk_id')[0];

            // Inactivar el actual
            InstrumentosAudPamec::on($this->cliente)->where('iap_pk_id', $idActivo)->update([
                'iap_activa' => false
            ]);

            HistoricoInsAudPamec::on($this->cliente)->create([
                'iap_fk_id' => $idActivo,
                'usu_fk_id' => $usuario,
                'hap_accion' => 'El instrumento pasa a estado inactivo'
            ]);

            // Activar el nuevo instrumento
            InstrumentosAudPamec::on($this->cliente)->where('iap_pk_id', $id)->update([
                'iap_activa' => true
            ]);

            HistoricoInsAudPamec::on($this->cliente)->create([
                'iap_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hap_accion' => 'El instrumento pasa a estado activo'
            ]);
        });
    }

    public function obtenerHistoricoVersion(Request $request, $id) {
        return HistoricoInsAudPamec::on($this->cliente)
                                   ->selectRaw('historico_ins_aud_pamec.*, users.name')
                                   ->join('base.users', 'id', 'usu_fk_id')
                                   ->where('iap_fk_id', $id)
                                   ->orderBy('created_at', 'desc')
                                   ->get()->toArray();
    }
}
